<?php
require_once __DIR__ . "/includes/admin_auth.php";

$adminUser = require_admin();
$pdo = db();

$pageTitle = "Diaporama";

$msg = null;
$err = null;

function bool_from_post(string $key): int {
  return isset($_POST[$key]) && ($_POST[$key] === "1" || $_POST[$key] === "on") ? 1 : 0;
}

function safe_url(?string $url): ?string {
  $url = trim((string)$url);
  if ($url === "") return null;
  if (!preg_match('#^https?://#i', $url)) return null;
  return filter_var($url, FILTER_VALIDATE_URL) ? $url : null;
}

$uploadRoot = realpath(__DIR__ . "/../uploads");
$uploadDir = __DIR__ . "/../uploads/carousel";
if (!is_dir($uploadDir)) @mkdir($uploadDir, 0775, true);

if ($_SERVER["REQUEST_METHOD"] === "POST") {
  $action = $_POST["action"] ?? "";

  if ($action === "upload") {
    if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
      $err = "Image requise (jpg/png/webp).";
    } else {
      $file = $_FILES['image'];
      $maxBytes = 2 * 1024 * 1024; // 2MB
      if ($file['size'] > $maxBytes) {
        $err = "Image trop lourde (max 2 Mo).";
      } else {
        $tmp = $file['tmp_name'];
        $info = @getimagesize($tmp);
        if (!$info) {
          $err = "Fichier invalide.";
        } else {
          $mime = $info['mime'] ?? '';
          $allowed = [
            'image/jpeg' => 'jpg',
            'image/png'  => 'png',
            'image/webp' => 'webp',
          ];
          if (!isset($allowed[$mime])) {
            $err = "Format non supporté. Utilisez JPG, PNG ou WEBP.";
          } else {
            $ext = $allowed[$mime];
            $name = "carousel_" . date('Ymd_His') . "_" . bin2hex(random_bytes(4)) . "." . $ext;
            $destFs = $uploadDir . "/" . $name;

            if (!@move_uploaded_file($tmp, $destFs)) {
              $err = "Impossible de sauvegarder l'image (droits dossier uploads?).";
            } else {
              $title = trim($_POST['title'] ?? "");
              $sortOrder = (int)($_POST['sort_order'] ?? 1);
              $sortOrder = max(1, min(99, $sortOrder));
              $isActive = bool_from_post('is_active');
              $isClickable = bool_from_post('is_clickable');
              $linkUrl = $isClickable ? safe_url($_POST['link_url'] ?? null) : null;

              if ($isClickable && !$linkUrl) {
                // Si cliquable demandé mais URL invalide, on désactive le clic
                $isClickable = 0;
              }

              $webPath = "uploads/carousel/" . $name;
              $stmt = $pdo->prepare("INSERT INTO carousel_images (title,image_path,link_url,is_clickable,is_active,sort_order) VALUES (?,?,?,?,?,?)");
              $stmt->execute([
                $title !== "" ? $title : null,
                $webPath,
                $linkUrl,
                $isClickable,
                $isActive,
                $sortOrder
              ]);
              $msg = "Image ajoutée au diaporama.";
            }
          }
        }
      }
    }
  }

  if ($action === "update") {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $title = trim($_POST['title'] ?? "");
      $sortOrder = (int)($_POST['sort_order'] ?? 1);
      $sortOrder = max(1, min(99, $sortOrder));
      $isActive = bool_from_post('is_active');
      $isClickable = bool_from_post('is_clickable');
      $linkUrl = $isClickable ? safe_url($_POST['link_url'] ?? null) : null;
      if ($isClickable && !$linkUrl) $isClickable = 0;

      $stmt = $pdo->prepare("UPDATE carousel_images SET title=?, link_url=?, is_clickable=?, is_active=?, sort_order=?, updated_at=NOW() WHERE id=?");
      $stmt->execute([
        $title !== "" ? $title : null,
        $linkUrl,
        $isClickable,
        $isActive,
        $sortOrder,
        $id
      ]);
      $msg = "Image mise à jour.";
    }
  }

  if ($action === "delete") {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $stmt = $pdo->prepare("SELECT image_path FROM carousel_images WHERE id=? LIMIT 1");
      $stmt->execute([$id]);
      $row = $stmt->fetch();
      if ($row) {
        $path = (string)$row['image_path'];
        // Supprimer fichier si dans uploads/
        $fs = realpath(__DIR__ . "/../" . $path);
        if ($fs && $uploadRoot && strpos($fs, $uploadRoot) === 0) {
          @unlink($fs);
        }
      }
      $stmt = $pdo->prepare("DELETE FROM carousel_images WHERE id=?");
      $stmt->execute([$id]);
      $msg = "Image supprimée.";
    }
  }
}

$items = $pdo->query("SELECT id,title,image_path,link_url,is_clickable,is_active,sort_order,updated_at FROM carousel_images ORDER BY sort_order ASC, id ASC")->fetchAll();

require __DIR__ . "/includes/header.php";
?>

<div class="row g-3">
  <div class="col-lg-5">
    <div class="card p-4">
      <h5 class="mb-2">Ajouter une image (diaporama)</h5>
      <div class="small-muted mb-3">
        Recommandé: <b>JPG ou WEBP</b>, format <b>16:9</b> (ex: 1280×720), max <b>2 Mo</b>.
      </div>

      <?php if ($msg): ?><div class="alert alert-success"><?= htmlspecialchars($msg) ?></div><?php endif; ?>
      <?php if ($err): ?><div class="alert alert-danger"><?= htmlspecialchars($err) ?></div><?php endif; ?>

      <form method="post" enctype="multipart/form-data" class="vstack gap-2">
        <input type="hidden" name="action" value="upload">

        <div>
          <label class="form-label">Image</label>
          <input class="form-control" type="file" name="image" accept="image/jpeg,image/png,image/webp" required>
        </div>

        <div class="row g-2">
          <div class="col-7">
            <label class="form-label">Titre (optionnel)</label>
            <input class="form-control" name="title" placeholder="ex: Révise ton Anglais" />
          </div>
          <div class="col-5">
            <label class="form-label">Ordre (1..99)</label>
            <input class="form-control" name="sort_order" type="number" min="1" max="99" value="1" />
          </div>
        </div>

        <div class="form-check form-switch">
          <input class="form-check-input" type="checkbox" role="switch" id="active" name="is_active" value="1" checked>
          <label class="form-check-label" for="active">Actif (affiché)</label>
        </div>

        <div class="form-check form-switch">
          <input class="form-check-input" type="checkbox" role="switch" id="clickable" name="is_clickable" value="1">
          <label class="form-check-label" for="clickable">Cliquable (ouvre un lien)</label>
        </div>

        <div>
          <label class="form-label">Lien (https://...) si cliquable</label>
          <input class="form-control" name="link_url" placeholder="https://..." />
        </div>

        <button class="btn btn-primary">Ajouter</button>

        <div class="small-muted">L'application affiche jusqu'à <b>3 images actives</b> (ordre croissant).</div>
      </form>
    </div>
  </div>

  <div class="col-lg-7">
    <div class="card p-4">
      <h5 class="mb-3">Images du diaporama</h5>
      <div class="table-responsive">
        <table class="table table-hover align-middle">
          <thead>
            <tr>
              <th>Prévisualisation</th>
              <th>Détails</th>
              <th class="text-end">Actions</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($items as $it): ?>
              <tr>
                <td style="width:170px;">
                  <img src="../<?= htmlspecialchars($it['image_path']) ?>" style="width:160px;height:90px;object-fit:cover;border-radius:14px;" alt="">
                </td>
                <td>
                  <div class="fw-semibold">#<?= (int)$it['id'] ?> — <?= htmlspecialchars($it['title'] ?? 'Sans titre') ?></div>
                  <div class="small-muted">Ordre: <?= (int)$it['sort_order'] ?> • <?= ((int)$it['is_active']===1) ? 'Actif' : 'Inactif' ?> • <?= ((int)$it['is_clickable']===1) ? 'Cliquable' : 'Non cliquable' ?></div>
                  <?php if (!empty($it['link_url'])): ?>
                    <div class="small"><a href="<?= htmlspecialchars($it['link_url']) ?>" target="_blank" rel="noopener">Ouvrir le lien</a></div>
                  <?php endif; ?>
                  <div class="small-muted">Maj: <?= htmlspecialchars($it['updated_at']) ?></div>
                </td>
                <td class="text-end">
                  <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="collapse" data-bs-target="#edit<?= (int)$it['id'] ?>">Modifier</button>

                  <form method="post" class="d-inline" onsubmit="return confirm('Supprimer cette image ?');">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?= (int)$it['id'] ?>">
                    <button class="btn btn-sm btn-outline-danger">Supprimer</button>
                  </form>

                  <div class="collapse mt-2" id="edit<?= (int)$it['id'] ?>">
                    <form method="post" class="card card-body border-0 bg-body-tertiary mt-2">
                      <input type="hidden" name="action" value="update">
                      <input type="hidden" name="id" value="<?= (int)$it['id'] ?>">

                      <div class="mb-2">
                        <label class="form-label">Titre</label>
                        <input class="form-control" name="title" value="<?= htmlspecialchars($it['title'] ?? '') ?>">
                      </div>

                      <div class="mb-2">
                        <label class="form-label">Ordre</label>
                        <input class="form-control" name="sort_order" type="number" min="1" max="99" value="<?= (int)$it['sort_order'] ?>">
                      </div>

                      <div class="form-check form-switch mb-2">
                        <input class="form-check-input" type="checkbox" name="is_active" value="1" <?= ((int)$it['is_active']===1) ? 'checked' : '' ?> >
                        <label class="form-check-label">Actif</label>
                      </div>

                      <div class="form-check form-switch mb-2">
                        <input class="form-check-input" type="checkbox" name="is_clickable" value="1" <?= ((int)$it['is_clickable']===1) ? 'checked' : '' ?> >
                        <label class="form-check-label">Cliquable</label>
                      </div>

                      <div class="mb-2">
                        <label class="form-label">Lien (https://...)</label>
                        <input class="form-control" name="link_url" value="<?= htmlspecialchars($it['link_url'] ?? '') ?>">
                      </div>

                      <button class="btn btn-primary btn-sm">Enregistrer</button>
                    </form>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>
            <?php if (count($items) === 0): ?>
              <tr><td colspan="3" class="text-center text-body-secondary">Aucune image. Ajoutez-en une à gauche.</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>

      <div class="small-muted mt-2">
        L'application prend les <b>3 premières</b> images actives (ordre + id). Désactivez une image pour la retirer sans la supprimer.
      </div>
    </div>
  </div>
</div>

<?php require __DIR__ . "/includes/footer.php"; ?>
