<?php
require_once __DIR__ . "/includes/admin_auth.php";
require_once __DIR__ . "/includes/meta.php";

$adminUser = require_admin();
$pdo = db();

$classes = get_classes($pdo);
$subjects = get_subjects($pdo);

// ==============================
// Téléchargements CSV (template / export)
// ==============================

// Télécharge un modèle CSV prêt à remplir
if (($_GET["download"] ?? "") === "template") {
  header('Content-Type: text/csv; charset=UTF-8');
  header('Content-Disposition: attachment; filename="exemple_questions_import.csv"');
  // BOM UTF-8 (utile pour Excel)
  echo "\xEF\xBB\xBF";
  $out = fopen('php://output', 'w');
  $sep = ';';
  fputcsv($out, [
    'class','subject','question','option_a','option_b','option_c','option_d','correct','explanation'
  ], $sep);
  // 2 lignes d'exemple (à remplacer)
  fputcsv($out, [
    'CM2','Français','Dans « Il mange », quel est le verbe ?','Il','mange','la phrase','aucun','B','Le verbe exprime l\'action.'
  ], $sep);
  fputcsv($out, [
    '6ème','SVT','Quel organe permet principalement la respiration ?','Le cœur','Les poumons','Le foie','Les reins','B','Les poumons assurent les échanges gazeux.'
  ], $sep);
  fclose($out);
  exit;
}

// Exporter les questions existantes en CSV (selon les filtres)
if (($_GET["download"] ?? "") === "export") {
  // Export potentiellement volumineux : éviter un timeout trop court
  @set_time_limit(0);
  $filterClassId = (int)($_GET["class_id"] ?? 0);
  $filterSubjectId = (int)($_GET["subject_id"] ?? 0);

  $sql = "SELECT q.question, q.option_a, q.option_b, q.option_c, q.option_d, q.correct_option, q.explanation,
                 c.name AS class_name, s.name AS subject_name
          FROM questions q
          JOIN classes c ON c.id = q.class_id
          JOIN subjects s ON s.id = q.subject_id
          WHERE 1=1";
  $params = [];
  if ($filterClassId > 0) { $sql .= " AND q.class_id = ?"; $params[] = $filterClassId; }
  if ($filterSubjectId > 0) { $sql .= " AND q.subject_id = ?"; $params[] = $filterSubjectId; }
  $sql .= " ORDER BY c.name ASC, s.name ASC, q.id ASC";

  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);

  $date = date('Y-m-d');
  header('Content-Type: text/csv; charset=UTF-8');
  header('Content-Disposition: attachment; filename="questions_export_' . $date . '.csv"');
  echo "\xEF\xBB\xBF";
  $out = fopen('php://output', 'w');
  $sep = ';';
  fputcsv($out, [
    'class','subject','question','option_a','option_b','option_c','option_d','correct','explanation'
  ], $sep);
  while ($row = $stmt->fetch()) {
    fputcsv($out, [
      (string)$row['class_name'],
      (string)$row['subject_name'],
      (string)$row['question'],
      (string)$row['option_a'],
      (string)$row['option_b'],
      (string)$row['option_c'],
      (string)$row['option_d'],
      (string)$row['correct_option'],
      (string)($row['explanation'] ?? ''),
    ], $sep);
  }
  fclose($out);
  exit;
}

$msg = null;
$err = null;
$importReport = null;

if ($_SERVER["REQUEST_METHOD"] === "POST") {
  $action = $_POST["action"] ?? "add";

  if ($action === "delete") {
    $id = (int)($_POST["id"] ?? 0);
    if ($id > 0) {
      $stmt = $pdo->prepare("DELETE FROM questions WHERE id = ?");
      $stmt->execute([$id]);
      $msg = "Question supprimée.";
    }
  } else if ($action === "add") {
    $classId = (int)($_POST["class_id"] ?? 0);
    $subjectId = (int)($_POST["subject_id"] ?? 0);
    $question = trim($_POST["question"] ?? "");
    $a = trim($_POST["a"] ?? "");
    $b = trim($_POST["b"] ?? "");
    $c = trim($_POST["c"] ?? "");
    $d = trim($_POST["d"] ?? "");
    $correct = strtoupper(trim($_POST["correct"] ?? ""));
    $explanation = trim($_POST["explanation"] ?? "");

    if ($classId <= 0 || $subjectId <= 0 || $question === "" || $a === "" || $b === "" || $c === "" || $d === "" || !in_array($correct, ["A","B","C","D"], true)) {
      $err = "Veuillez remplir tous les champs requis.";
    } else {
      $stmt = $pdo->prepare(
        "INSERT INTO questions (class_id, subject_id, question, option_a, option_b, option_c, option_d, correct_option, explanation, created_at)
         VALUES (?,?,?,?,?,?,?,?,?,NOW())"
      );
      $stmt->execute([$classId,$subjectId,$question,$a,$b,$c,$d,$correct,$explanation !== "" ? $explanation : null]);
      $msg = "Question ajoutée.";
    }
  } else if ($action === "import_csv") {
    // Import en masse via CSV (UTF-8). En-têtes supportés (insensible à la casse) :
    // class / classe, subject / filiere / filière, question, option_a/a, option_b/b, option_c/c, option_d/d,
    // correct / correct_option, explanation (optionnel)

    if (!isset($_FILES["csv"]) || ($_FILES["csv"]["error"] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
      $err = "Veuillez choisir un fichier CSV valide.";
    } else {
      $tmp = $_FILES["csv"]["tmp_name"];
      $fh = fopen($tmp, "r");
      if (!$fh) {
        $err = "Impossible de lire le fichier CSV.";
      } else {
        // Détection simple du séparateur (',' ou ';') à partir de la première ligne
        $firstLine = fgets($fh);
        if ($firstLine === false) {
          fclose($fh);
          $err = "Fichier CSV vide.";
        } else {
          $sep = (substr_count($firstLine, ";") > substr_count($firstLine, ",")) ? ";" : ",";
          rewind($fh);

          // Helper : normaliser un nom de colonne
          $norm = function(string $s): string {
            $s = trim($s);
            // retirer BOM éventuel
            $s = preg_replace('/^\xEF\xBB\xBF/', '', $s);
            $s = mb_strtolower($s);
            $s = str_replace([" ","-"], ["_","_"], $s);
            $s = str_replace(["é","è","ê","ë"], "e", $s);
            $s = str_replace(["à","â"], "a", $s);
            $s = str_replace(["î","ï"], "i", $s);
            $s = str_replace(["ô","ö"], "o", $s);
            $s = str_replace(["ù","û","ü"], "u", $s);
            $s = str_replace(["ç"], "c", $s);
            return $s;
          };

          $header = fgetcsv($fh, 0, $sep);
          if (!$header) {
            fclose($fh);
            $err = "Impossible de lire l'en-tête du CSV.";
          } else {
            $colIndex = [];
            foreach ($header as $i => $h) {
              $key = $norm((string)$h);
              if ($key !== "") $colIndex[$key] = $i;
            }

            // Map des alias vers clés attendues
            $getIdx = function(array $aliases) use ($colIndex) {
              foreach ($aliases as $a) {
                if (array_key_exists($a, $colIndex)) return $colIndex[$a];
              }
              return null;
            };

            $idxClass = $getIdx(["class","classe","class_name"]);
            $idxSubject = $getIdx(["subject","filiere","filiere_name","matiere","discipline"]);
            $idxQuestion = $getIdx(["question","q"]);
            $idxA = $getIdx(["option_a","a"]);
            $idxB = $getIdx(["option_b","b"]);
            $idxC = $getIdx(["option_c","c"]);
            $idxD = $getIdx(["option_d","d"]);
            $idxCorrect = $getIdx(["correct","correct_option","reponse","answer"]);
            $idxExplain = $getIdx(["explanation","explication","explain"]);

            $missing = [];
            if ($idxClass === null) $missing[] = "class";
            if ($idxSubject === null) $missing[] = "subject";
            if ($idxQuestion === null) $missing[] = "question";
            if ($idxA === null) $missing[] = "option_a";
            if ($idxB === null) $missing[] = "option_b";
            if ($idxC === null) $missing[] = "option_c";
            if ($idxD === null) $missing[] = "option_d";
            if ($idxCorrect === null) $missing[] = "correct";
            if (!empty($missing)) {
              fclose($fh);
              $err = "Colonnes manquantes dans le CSV: " . implode(", ", $missing) . ".";
            } else {
              // Construire des maps nom->id (insensible à la casse)
              $classMap = [];
              foreach ($classes as $c) {
                $classMap[mb_strtolower($c["name"])] = (int)$c["id"];
              }
              $subjectMap = [];
              foreach ($subjects as $s) {
                $subjectMap[mb_strtolower($s["name"])] = (int)$s["id"];
              }

              $rows = [];
              $errors = [];
              $line = 1; // header = 1
              while (($data = fgetcsv($fh, 0, $sep)) !== false) {
                $line++;
                // Ignorer lignes vides
                $allEmpty = true;
                foreach ($data as $v) { if (trim((string)$v) !== "") { $allEmpty = false; break; } }
                if ($allEmpty) continue;

                $className = trim((string)($data[$idxClass] ?? ""));
                $subjectName = trim((string)($data[$idxSubject] ?? ""));
                $question = trim((string)($data[$idxQuestion] ?? ""));
                $a = trim((string)($data[$idxA] ?? ""));
                $b = trim((string)($data[$idxB] ?? ""));
                $c = trim((string)($data[$idxC] ?? ""));
                $d = trim((string)($data[$idxD] ?? ""));
                $correct = strtoupper(trim((string)($data[$idxCorrect] ?? "")));
                $explanation = $idxExplain !== null ? trim((string)($data[$idxExplain] ?? "")) : "";

                $classId = $classMap[mb_strtolower($className)] ?? 0;
                $subjectId = $subjectMap[mb_strtolower($subjectName)] ?? 0;

                $rowErr = [];
                if ($classId <= 0) $rowErr[] = "Classe inconnue: '$className'";
                if ($subjectId <= 0) $rowErr[] = "Filière inconnue: '$subjectName'";
                if ($question === "") $rowErr[] = "Question vide";
                if ($a === "" || $b === "" || $c === "" || $d === "") $rowErr[] = "Options A/B/C/D incomplètes";
                if (!in_array($correct, ["A","B","C","D"], true)) $rowErr[] = "Correct doit être A, B, C ou D";

                if (!empty($rowErr)) {
                  $errors[] = "Ligne $line: " . implode(" | ", $rowErr);
                } else {
                  $rows[] = [$classId,$subjectId,$question,$a,$b,$c,$d,$correct,$explanation !== "" ? $explanation : null];
                }
              }
              fclose($fh);

              if (empty($rows) && empty($errors)) {
                $err = "Aucune ligne de données détectée (vérifiez votre CSV).";
              } else if (!empty($errors)) {
                // On n'importe pas si des erreurs existent (plus simple/sûr pour débutant)
                $importReport = [
                  "imported" => 0,
                  "total" => count($rows) + count($errors),
                  "errors" => $errors,
                ];
                $err = "Import annulé: corrigez les erreurs ci-dessous puis réessayez.";
              } else {
                // Import transactionnel
                $pdo->beginTransaction();
                try {
                  $stmt = $pdo->prepare(
                    "INSERT INTO questions (class_id, subject_id, question, option_a, option_b, option_c, option_d, correct_option, explanation, created_at)
                     VALUES (?,?,?,?,?,?,?,?,?,NOW())"
                  );
                  foreach ($rows as $r) {
                    $stmt->execute($r);
                  }
                  $pdo->commit();
                  $importReport = [
                    "imported" => count($rows),
                    "total" => count($rows),
                    "errors" => [],
                  ];
                  $msg = "Import terminé: " . count($rows) . " question(s) ajoutée(s).";
                } catch (Throwable $t) {
                  $pdo->rollBack();
                  $err = "Erreur pendant l'import: " . $t->getMessage();
                }
              }
            }
          }
        }
      }
    }
  }
}

$filterClassId = (int)($_GET["class_id"] ?? 0);
$filterSubjectId = (int)($_GET["subject_id"] ?? 0);

$sql = "SELECT q.id, q.question, q.correct_option, c.name AS class_name, s.name AS subject_name
        FROM questions q
        JOIN classes c ON c.id = q.class_id
        JOIN subjects s ON s.id = q.subject_id
        WHERE 1=1";
$params = [];
if ($filterClassId > 0) { $sql .= " AND q.class_id = ?"; $params[] = $filterClassId; }
if ($filterSubjectId > 0) { $sql .= " AND q.subject_id = ?"; $params[] = $filterSubjectId; }
$sql .= " ORDER BY q.id DESC LIMIT 200";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$questions = $stmt->fetchAll();

$pageTitle = "Questions";
require __DIR__ . "/includes/header.php";
?>

<div class="card p-4 mb-3">
  <h5 class="mb-3">Ajouter une question</h5>

  <?php if ($msg): ?><div class="alert alert-success"><?= htmlspecialchars($msg) ?></div><?php endif; ?>
  <?php if ($err): ?><div class="alert alert-danger"><?= htmlspecialchars($err) ?></div><?php endif; ?>

  <form method="post" class="row g-3">
    <input type="hidden" name="action" value="add">
    <div class="col-md-4">
      <label class="form-label">Classe</label>
      <select class="form-select" name="class_id" required>
        <?php foreach ($classes as $c): ?>
          <option value="<?= (int)$c["id"] ?>"><?= htmlspecialchars($c["name"]) ?></option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-md-4">
      <label class="form-label">Filière</label>
      <select class="form-select" name="subject_id" required>
        <?php foreach ($subjects as $s): ?>
          <option value="<?= (int)$s["id"] ?>"><?= htmlspecialchars($s["name"]) ?></option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-md-4">
      <label class="form-label">Réponse correcte</label>
      <select class="form-select" name="correct" required>
        <option value="A">A</option>
        <option value="B">B</option>
        <option value="C">C</option>
        <option value="D">D</option>
      </select>
    </div>
    <div class="col-12">
      <label class="form-label">Question</label>
      <textarea class="form-control" name="question" rows="2" required></textarea>
    </div>
    <div class="col-md-6">
      <label class="form-label">Option A</label>
      <input class="form-control" name="a" required>
    </div>
    <div class="col-md-6">
      <label class="form-label">Option B</label>
      <input class="form-control" name="b" required>
    </div>
    <div class="col-md-6">
      <label class="form-label">Option C</label>
      <input class="form-control" name="c" required>
    </div>
    <div class="col-md-6">
      <label class="form-label">Option D</label>
      <input class="form-control" name="d" required>
    </div>
    <div class="col-12">
      <label class="form-label">Explication (optionnel)</label>
      <textarea class="form-control" name="explanation" rows="2" placeholder="Pourquoi la réponse est correcte ?"></textarea>
    </div>
    <div class="col-12">
      <button class="btn btn-primary">Ajouter</button>
      <span class="ms-2 small-muted">Conseil: créez au moins 20 questions par filière et par classe.</span>
    </div>
  </form>
</div>

<div class="card p-4 mb-3">
  <div class="d-flex flex-wrap justify-content-between align-items-center gap-2 mb-2">
    <h5 class="mb-0">Importer des questions en masse (CSV)</h5>
    <a class="btn btn-sm btn-outline-secondary" href="questions.php?download=template">Télécharger un exemple CSV</a>
  </div>
  <div class="small-muted mb-3">
    Format CSV (UTF-8) avec en-tête. Séparateur accepté: <b>;</b> ou <b>,</b>.<br>
    Colonnes obligatoires: <code>class</code>, <code>subject</code>, <code>question</code>, <code>option_a</code>, <code>option_b</code>, <code>option_c</code>, <code>option_d</code>, <code>correct</code>.
    Colonne optionnelle: <code>explanation</code>.<br>
    <span class="text-body-secondary">Exemple de 1 ligne:</span>
    <code>CM2;Français;Dans « Il mange », quel est le verbe ?;Il;mange;la phrase;aucun;B;Le verbe exprime l'action.</code>
  </div>

  <?php if ($importReport && !empty($importReport["errors"])): ?>
    <div class="alert alert-warning">
      <div class="fw-semibold mb-2">Erreurs détectées (corrigez puis réessayez) :</div>
      <ul class="mb-0">
        <?php foreach (array_slice($importReport["errors"], 0, 20) as $e): ?>
          <li><?= htmlspecialchars($e) ?></li>
        <?php endforeach; ?>
      </ul>
      <?php if (count($importReport["errors"]) > 20): ?>
        <div class="small-muted mt-2">Affichage limité à 20 erreurs (sur <?= (int)count($importReport["errors"]) ?>).</div>
      <?php endif; ?>
    </div>
  <?php endif; ?>

  <form method="post" enctype="multipart/form-data" class="row g-3">
    <input type="hidden" name="action" value="import_csv">
    <div class="col-md-8">
      <input class="form-control" type="file" name="csv" accept=".csv,text/csv" required>
    </div>
    <div class="col-md-4">
      <button class="btn btn-outline-primary w-100">Importer</button>
    </div>
  </form>

  <div class="small-muted mt-3">
    Astuce: crée d'abord tes <b>Classes</b> et <b>Filières</b> dans les menus correspondants. Le CSV doit utiliser exactement les mêmes noms.
  </div>
</div>

<div class="card p-4">
  <div class="d-flex flex-wrap justify-content-between align-items-center gap-2 mb-2">
    <h5 class="mb-0">Liste des questions</h5>
    <div class="d-flex flex-wrap align-items-center gap-2">
      <a class="btn btn-outline-secondary btn-sm" href="questions.php?download=export&class_id=<?= (int)$filterClassId ?>&subject_id=<?= (int)$filterSubjectId ?>">Exporter en CSV</a>
      <form class="d-flex gap-2" method="get">
      <select class="form-select form-select-sm" name="class_id">
        <option value="0">Toutes classes</option>
        <?php foreach ($classes as $c): ?>
          <option value="<?= (int)$c["id"] ?>" <?= $filterClassId===(int)$c["id"] ? "selected" : "" ?>><?= htmlspecialchars($c["name"]) ?></option>
        <?php endforeach; ?>
      </select>
      <select class="form-select form-select-sm" name="subject_id">
        <option value="0">Toutes filières</option>
        <?php foreach ($subjects as $s): ?>
          <option value="<?= (int)$s["id"] ?>" <?= $filterSubjectId===(int)$s["id"] ? "selected" : "" ?>><?= htmlspecialchars($s["name"]) ?></option>
        <?php endforeach; ?>
      </select>
      <button class="btn btn-outline-primary btn-sm">Filtrer</button>
      </form>
    </div>
  </div>

  <div class="table-responsive">
    <table class="table table-hover align-middle">
      <thead>
        <tr>
          <th>ID</th>
          <th>Classe</th>
          <th>Filière</th>
          <th>Question</th>
          <th>Correct</th>
          <th class="text-end">Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($questions as $q): ?>
        <tr>
          <td><?= (int)$q["id"] ?></td>
          <td><span class="badge badge-soft"><?= htmlspecialchars($q["class_name"]) ?></span></td>
          <td><?= htmlspecialchars($q["subject_name"]) ?></td>
          <td><?= htmlspecialchars(mb_strimwidth($q["question"], 0, 80, "…")) ?></td>
          <td class="fw-semibold"><?= htmlspecialchars($q["correct_option"]) ?></td>
          <td class="text-end">
            <form method="post" onsubmit="return confirm('Supprimer cette question ?');" style="display:inline;">
              <input type="hidden" name="action" value="delete">
              <input type="hidden" name="id" value="<?= (int)$q["id"] ?>">
              <button class="btn btn-sm btn-outline-danger">Supprimer</button>
            </form>
          </td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
  <div class="small-muted mt-2">Affichage limité à 200 dernières questions.</div>
</div>

<?php require __DIR__ . "/includes/footer.php"; ?>
