<?php
require_once __DIR__ . "/helpers.php";
require_once __DIR__ . "/db.php";
require_once __DIR__ . "/auth.php";

cors();

$path = parse_url($_SERVER["REQUEST_URI"], PHP_URL_PATH);
$scriptDir = rtrim(dirname($_SERVER["SCRIPT_NAME"]), "/"); // e.g. /api
$endpoint = trim(substr($path, strlen($scriptDir)), "/");

function read_class_id(PDO $pdo, array $data): int {
    // v2 prefers classId. For backward-compat, accept "classe" (name) too.
    if (isset($data["classId"])) {
        $id = (int)$data["classId"];
        if ($id <= 0) json_response(["error" => "classId invalide"], 400);
        $stmt = $pdo->prepare("SELECT id FROM classes WHERE id = ? AND is_active=1 LIMIT 1");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) json_response(["error" => "Classe inconnue"], 400);
        return $id;
    }

    if (isset($data["classe"])) {
        $name = trim((string)$data["classe"]);
        if ($name === "") json_response(["error" => "classe invalide"], 400);
        $stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ? AND is_active=1 LIMIT 1");
        $stmt->execute([$name]);
        $row = $stmt->fetch();
        if (!$row) json_response(["error" => "Classe inconnue"], 400);
        return (int)$row["id"];
    }

    json_response(["error" => "classId requis"], 400);
}

function read_subject_id(PDO $pdo, $raw): int {
    if ($raw === null) json_response(["error" => "subjectId requis"], 400);
    $id = (int)$raw;
    if ($id <= 0) json_response(["error" => "subjectId invalide"], 400);
    $stmt = $pdo->prepare("SELECT id FROM subjects WHERE id = ? AND is_active=1 LIMIT 1");
    $stmt->execute([$id]);
    if (!$stmt->fetch()) json_response(["error" => "Filière inconnue"], 400);
    return $id;
}

try {
    switch ($endpoint) {
        case "":
        case "health":
            json_response(["ok" => true, "service" => "QuizCM API", "version" => "v2"]);
            break;

        // ---- Public meta endpoints ----
        case "classes":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $pdo = db();
            // Tri côté serveur: ordre choisi dans l'Admin (sort_order), puis nom.
            $rows = $pdo->query("SELECT id,name FROM classes WHERE is_active=1 ORDER BY sort_order ASC, name ASC")->fetchAll();
            $items = array_map(fn($r) => ["id" => (int)$r["id"], "name" => $r["name"]], $rows);
            json_response(["items" => $items]);
            break;

        case "subjects":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $pdo = db();
            // Tri côté serveur: ordre choisi dans l'Admin (sort_order), puis nom.
            $rows = $pdo->query("SELECT id,name FROM subjects WHERE is_active=1 ORDER BY sort_order ASC, name ASC")->fetchAll();
            $items = array_map(fn($r) => ["id" => (int)$r["id"], "name" => $r["name"]], $rows);
            json_response(["items" => $items]);
            break;

        case "settings":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $pdo = db();
            $quizTime = (int)get_setting($pdo, "quiz_time_seconds", "20");
            if ($quizTime < 5) $quizTime = 5;
            if ($quizTime > 600) $quizTime = 600;

            $carouselInterval = (int)get_setting($pdo, "carousel_interval_seconds", "5");
            if ($carouselInterval < 2) $carouselInterval = 2;
            if ($carouselInterval > 60) $carouselInterval = 60;

            // --- Ads (AdMob) ---
            $adsEnabled = get_setting($pdo, "ads_enabled", "0") === "1";
            $bannerEnabled = get_setting($pdo, "banner_enabled", "0") === "1";
            $interstitialEnabled = get_setting($pdo, "interstitial_enabled", "0") === "1";
            $bannerUnitId = trim(get_setting($pdo, "admob_banner_unit_id", ""));
            $interstitialUnitId = trim(get_setting($pdo, "admob_interstitial_unit_id", ""));
            $interstitialEvery = (int)get_setting($pdo, "interstitial_every", "1");
            $interstitialCooldown = (int)get_setting($pdo, "interstitial_cooldown_seconds", "30");
            $childDirected = get_setting($pdo, "ads_child_directed", "1") === "1";
            $nonPersonalized = get_setting($pdo, "ads_non_personalized", "1") === "1";
            $consentEnabled = get_setting($pdo, "ads_consent_enabled", "1") === "1";

            $interstitialEvery = max(1, min(50, $interstitialEvery));
            $interstitialCooldown = max(0, min(3600, $interstitialCooldown));

            json_response([
                "quizTimeSeconds" => $quizTime,
                "carouselIntervalSeconds" => $carouselInterval,
                "ads" => [
                    "enabled" => $adsEnabled,
                    "bannerEnabled" => $bannerEnabled,
                    "interstitialEnabled" => $interstitialEnabled,
                    "bannerAdUnitId" => $bannerUnitId !== "" ? $bannerUnitId : null,
                    "interstitialAdUnitId" => $interstitialUnitId !== "" ? $interstitialUnitId : null,
                    "interstitialEvery" => $interstitialEvery,
                    "interstitialCooldownSeconds" => $interstitialCooldown,
                    "childDirected" => $childDirected,
                    "nonPersonalized" => $nonPersonalized,
                    "consentEnabled" => $consentEnabled,
                ]
            ]);
            break;

        case "carousel":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $pdo = db();
            // On renvoie jusqu'à 3 images actives, ordonnées
            $stmt = $pdo->prepare("SELECT id,title,image_path,link_url,is_clickable,sort_order FROM carousel_images WHERE is_active=1 ORDER BY sort_order ASC, id ASC LIMIT 3");
            $stmt->execute();
            $rows = $stmt->fetchAll();

            $xf = $_SERVER['HTTP_X_FORWARDED_PROTO'] ?? null;
            $scheme = $xf ? $xf : ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http');
            $host = $_SERVER['HTTP_HOST'] ?? '';
            $base = $host ? ($scheme . '://' . $host) : '';

            $items = [];
            foreach ($rows as $r) {
                $path = ltrim((string)$r['image_path'], '/');
                $imageUrl = $base ? ($base . '/' . $path) : ('/' . $path);
                $items[] = [
                    'id' => (int)$r['id'],
                    'title' => $r['title'] ? $r['title'] : null,
                    'imageUrl' => $imageUrl,
                    'isClickable' => ((int)$r['is_clickable'] === 1) && !empty($r['link_url']),
                    'linkUrl' => $r['link_url'] ? $r['link_url'] : null,
                    'sortOrder' => (int)$r['sort_order'],
                ];
            }

            json_response(['items' => $items]);
            break;

        // ---- Auth ----
        case "register":
            if ($_SERVER["REQUEST_METHOD"] !== "POST") json_response(["error" => "Méthode invalide"], 405);
            $data = get_json_body();
            require_fields($data, ["email", "password", "displayName", "phone"]);

            $email = strtolower(trim($data["email"]));
            $displayName = trim($data["displayName"]);
            $password = (string)$data["password"];
            $phoneRaw = trim((string)$data['phone']);
            $phone = preg_replace('/[^0-9+]/', '', $phoneRaw);
            if (strlen($phone) < 6) json_response(['error' => 'Numéro de téléphone invalide'], 400);
            if (strlen($phone) > 30) $phone = substr($phone, 0, 30);

            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) json_response(["error" => "Email invalide"], 400);
            if (strlen($password) < 6) json_response(["error" => "Mot de passe trop court (min 6)"], 400);

            $pdo = db();
            $classId = read_class_id($pdo, $data);

            // Points de départ (settings -> fallback config)
            $cfg = require __DIR__ . "/config.php";
            $defaultPoints = (int)get_setting($pdo, "default_points", (string)($cfg["default_points"] ?? 3));
            if ($defaultPoints < 0) $defaultPoints = 0;
            if ($defaultPoints > 9999) $defaultPoints = 9999;

            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            if ($stmt->fetch()) json_response(["error" => "Email déjà utilisé"], 409);

            $stmt = $pdo->prepare("SELECT id FROM users WHERE phone = ? LIMIT 1");
            $stmt->execute([$phone]);
            if ($stmt->fetch()) json_response(["error" => "Téléphone déjà utilisé"], 409);

            $hash = password_hash($password, PASSWORD_BCRYPT);
            $token = random_token(24);

            $stmt = $pdo->prepare(
                "INSERT INTO users (email, phone, password_hash, display_name, class_id, points, role, token, created_at, updated_at, token_created_at)
                 VALUES (?, ?, ?, ?, ?, ?, 'student', ?, NOW(), NOW(), NOW())"
            );
            $stmt->execute([$email, $phone, $hash, $displayName, $classId, $defaultPoints, $token]);

            $id = (int)$pdo->lastInsertId();
            $stmt = $pdo->prepare(
                "SELECT u.id,u.email,u.phone,u.display_name,u.class_id,c.name AS class_name,u.points,u.role,u.token
                 FROM users u JOIN classes c ON c.id=u.class_id WHERE u.id=? LIMIT 1"
            );
            $stmt->execute([$id]);
            $u = $stmt->fetch();

            json_response([
                "token" => $token,
                "user" => user_dto($u)
            ]);
            break;

        case "login":
            if ($_SERVER["REQUEST_METHOD"] !== "POST") json_response(["error" => "Méthode invalide"], 405);
            $data = get_json_body();
            require_fields($data, ["email", "password"]);

            $email = strtolower(trim($data["email"]));
            $password = (string)$data["password"];

            $pdo = db();
            $stmt = $pdo->prepare(
                "SELECT u.id,u.email,u.phone,u.display_name,u.class_id,c.name AS class_name,c.is_active AS class_active,u.points,u.role,u.is_blocked,u.password_hash
                 FROM users u JOIN classes c ON c.id=u.class_id
                 WHERE u.email = ? LIMIT 1"
            );
            $stmt->execute([$email]);
            $u = $stmt->fetch();
            if (!$u || !password_verify($password, $u["password_hash"])) {
                json_response(["error" => "Identifiants incorrects"], 401);
            }

            if ((int)($u['is_blocked'] ?? 0) === 1) {
                json_response(["error" => "Compte bloqué. Contactez l'administrateur."], 403);
            }

            if ((int)($u['class_active'] ?? 1) === 0) {
                json_response(["error" => "Classe désactivée. Contactez l'administrateur."], 403);
            }

            $token = random_token(24);
            $stmt = $pdo->prepare("UPDATE users SET token = ?, token_created_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$token, $u["id"]]);

            $u["token"] = $token;
            json_response([
                "token" => $token,
                "user" => user_dto($u)
            ]);
            break;

        case "me":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $u = require_user();
            json_response(user_dto($u));
            break;

        case "update_class":
            if ($_SERVER["REQUEST_METHOD"] !== "POST") json_response(["error" => "Méthode invalide"], 405);
            $u = require_user();
            $data = get_json_body();
            $pdo = db();
            $classId = read_class_id($pdo, $data);

            $stmt = $pdo->prepare("UPDATE users SET class_id=?, updated_at=NOW() WHERE id=?");
            $stmt->execute([$classId, (int)$u['id']]);

            $stmt = $pdo->prepare("SELECT u.id,u.email,u.phone,u.display_name,u.class_id,c.name AS class_name,u.points,u.role,u.token FROM users u JOIN classes c ON c.id=u.class_id WHERE u.id=? LIMIT 1");
            $stmt->execute([(int)$u['id']]);
            $nu = $stmt->fetch();
            if (!$nu) json_response(["error" => "Utilisateur introuvable"], 404);

            json_response(user_dto($nu));
            break;

        // ---- Quiz ----
        case "questions":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            $u = require_user();
            $pdo = db();

            $subjectId = read_subject_id($pdo, $_GET["subjectId"] ?? null);

            $limit = isset($_GET["limit"]) ? (int)$_GET["limit"] : 10;
            $limit = max(1, min(50, $limit));

            $stmt = $pdo->prepare(
                "SELECT q.id, q.question, q.option_a, q.option_b, q.option_c, q.option_d
                 FROM questions q
                 JOIN classes c ON c.id=q.class_id AND c.is_active=1
                 JOIN subjects s ON s.id=q.subject_id AND s.is_active=1
                 WHERE q.class_id = ? AND q.subject_id = ?
                 ORDER BY RAND()
                 LIMIT $limit"
            );
            $stmt->execute([(int)$u["class_id"], $subjectId]);

            $items = [];
            foreach ($stmt->fetchAll() as $q) {
                $items[] = [
                    "id" => (int)$q["id"],
                    "question" => $q["question"],
                    "optionA" => $q["option_a"],
                    "optionB" => $q["option_b"],
                    "optionC" => $q["option_c"],
                    "optionD" => $q["option_d"],
                ];
            }
            json_response(["items" => $items]);
            break;

        case "answer":
            if ($_SERVER["REQUEST_METHOD"] !== "POST") json_response(["error" => "Méthode invalide"], 405);
            $u = require_user();
            $data = get_json_body();
            require_fields($data, ["questionId", "chosen"]);

            $questionId = (int)$data["questionId"];
            $chosen = strtoupper(trim((string)$data["chosen"]));
            // 'X' means timeout (always wrong).
            if (!in_array($chosen, ["A","B","C","D","X"], true)) json_response(["error" => "Choix invalide"], 400);

            $pdo = db();
            $stmt = $pdo->prepare("SELECT q.id, q.correct_option, q.explanation, q.class_id, q.subject_id FROM questions q JOIN classes c ON c.id=q.class_id AND c.is_active=1 JOIN subjects s ON s.id=q.subject_id AND s.is_active=1 WHERE q.id = ? LIMIT 1");
            $stmt->execute([$questionId]);
            $q = $stmt->fetch();
            if (!$q) json_response(["error" => "Question introuvable"], 404);

            if ((int)$q["class_id"] !== (int)$u["class_id"]) json_response(["error" => "Question non autorisée pour votre classe"], 403);

            $correct = ($chosen !== "X") && (strtoupper($q["correct_option"]) === $chosen);
            $delta = $correct ? 1 : -1;

            $oldPoints = (int)$u["points"];
            $newPoints = $oldPoints + $delta;
            if ($newPoints < 0) $newPoints = 0;

            $stmt = $pdo->prepare("UPDATE users SET points = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$newPoints, (int)$u["id"]]);

            // Nouvelle règle: si l'utilisateur se trompe (ou timeout),
            // on enlève 1 point (min 0) ET le quiz recommence.
            $restart = !$correct;

            json_response([
                "correct" => $correct,
                "delta" => $delta,
                "points" => $newPoints,
                "restart" => $restart,
                "explanation" => $q["explanation"] ? $q["explanation"] : null
            ]);
            break;

        case "report":
            if ($_SERVER["REQUEST_METHOD"] !== "POST") json_response(["error" => "Méthode invalide"], 405);
            $u = require_user();
            $data = get_json_body();
            require_fields($data, ["questionId", "message"]);

            $questionId = (int)$data["questionId"];
            $message = trim((string)$data["message"]);
            if ($questionId <= 0) json_response(["error" => "questionId invalide"], 400);
            if (strlen($message) < 3) json_response(["error" => "Message trop court"], 400);
            if (strlen($message) > 2000) $message = substr($message, 0, 2000);

            $pdo = db();
            $stmt = $pdo->prepare("SELECT q.id, q.question, q.class_id, q.subject_id FROM questions q JOIN classes c ON c.id=q.class_id AND c.is_active=1 JOIN subjects s ON s.id=q.subject_id AND s.is_active=1 WHERE q.id = ? LIMIT 1");
            $stmt->execute([$questionId]);
            $q = $stmt->fetch();
            if (!$q) json_response(["error" => "Question introuvable"], 404);
            if ((int)$q["class_id"] !== (int)$u["class_id"]) json_response(["error" => "Question non autorisée"], 403);

            $stmt = $pdo->prepare("INSERT INTO reports (user_id, question_id, class_id, subject_id, question_text, message, is_resolved, created_at) VALUES (?, ?, ?, ?, ?, ?, 0, NOW())");
            $stmt->execute([(int)$u["id"], (int)$q["id"], (int)$q["class_id"], (int)$q["subject_id"], (string)$q["question"], $message]);

            json_response(["ok" => true]);
            break;

        case "leaderboard":
            if ($_SERVER["REQUEST_METHOD"] !== "GET") json_response(["error" => "Méthode invalide"], 405);
            require_user();

            $limit = isset($_GET["limit"]) ? (int)$_GET["limit"] : 50;
            $limit = max(1, min(200, $limit));

            $pdo = db();
            $stmt = $pdo->prepare(
                "SELECT u.display_name,u.email,u.points,c.name AS class_name
                 FROM users u JOIN classes c ON c.id=u.class_id
                 WHERE u.role = 'student' AND u.is_blocked = 0 AND c.is_active = 1
                 ORDER BY u.points DESC, u.id ASC
                 LIMIT $limit"
            );
            $stmt->execute();

            $items = [];
            $rank = 1;
            foreach ($stmt->fetchAll() as $row) {
                $items[] = [
                    "rank" => $rank++,
                    "displayName" => $row["display_name"],
                    "email" => mask_email($row["email"]),
                    "points" => (int)$row["points"],
                    "classe" => $row["class_name"],
                ];
            }

            json_response(["items" => $items]);
            break;

        default:
            json_response(["error" => "Endpoint introuvable"], 404);
    }
} catch (Throwable $e) {
    json_response(["error" => "Erreur serveur", "detail" => $e->getMessage()], 500);
}
